
# life.py
# based on https://simplegametutorials.github.io/pygamezero/life/
# info: https://en.wikipedia.org/wiki/Conway%27s_Game_of_Life

'''
Transitions:

    Any live cell with fewer than two live neighbours dies.
    Any live cell with two or three live neighbours lives on.
    Any live cell with more than three live neighbours dies.
    Any dead cell with exactly three live neighbours becomes a live cell.

 Input: a glider

      x
       x
     xxx
'''

import math
import os
os.environ['PYGAME_HIDE_SUPPORT_PROMPT'] = "hide"
import pygame
from pygame.locals import *

# some colors
BLACK = (   0,   0,   0)
WHITE = ( 255, 255, 255)
GRAY = ( 220, 220, 220 )
RED   = ( 255,   0,   0)
GREEN = (   0, 255,   0)
BLUE  = ( 0,   0,   255)
YELLOW  = ( 0,   255,   255)

NUM_COLS = 70
NUM_ROWS = 50
CELL_SIZE = 15



def toggleCell(x, y):
  xSel = min(math.floor(x/CELL_SIZE), NUM_COLS-1)
  ySel = min(math.floor(y/CELL_SIZE), NUM_ROWS-1)
  grid[ySel][xSel] = not grid[ySel][xSel]



def updateGrid():
  for y in range(NUM_ROWS):
    for x in range(NUM_COLS):
      numNeighs = countLiveNeighbors(x, y)

      # live cell with fewer than two live neighbours dies.
      if grid[y][x] and (numNeighs < 2):
        grid2[y][x] = False
    
      # live cell with two or three live neighbours lives on.
      elif grid[y][x] and  \
           (numNeighs == 2 or numNeighs == 3):
        grid2[y][x] = True
      
      # live cell with more than three live neighbours dies.
      elif grid[y][x] and (numNeighs > 3):
        grid2[y][x] = False
    
    
      # dead cell with three live neighbours becomes alive
      elif (not grid[y][x]) and (numNeighs == 3):
        grid2[y][x] = True

  for y in range(NUM_ROWS):
    for x in range(NUM_COLS):
      grid[y][x] = grid2[y][x]


def countLiveNeighbors(x, y):
  numNeighs = 0
  for dy in range(-1, 2):
    for dx in range(-1, 2):
      xn = (x+dx) % NUM_COLS
      yn = (y+dy) % NUM_ROWS
      if xn == x and yn == y:
        pass
      elif grid[yn][xn]:
        numNeighs += 1
  return numNeighs


def drawGrid(screen):
  screen.fill(WHITE)
  for y in range(NUM_ROWS):
    for x in range(NUM_COLS):
      if grid[y][x]:
        color = WHITE
      else:
        color = BLACK
   
      pygame.draw.rect(screen, color,
         (x*CELL_SIZE, y*CELL_SIZE, 
          CELL_SIZE-1, CELL_SIZE-1), 0)
   


# ---------- main -------------

# grid[y][x]
grid = [[False for x in range(NUM_COLS)] for y in range(NUM_ROWS)]
grid2 = [[False for x in range(NUM_COLS)] for y in range(NUM_ROWS)]

pygame.init()
screen = pygame.display.set_mode([NUM_COLS*CELL_SIZE,
                                  NUM_ROWS*CELL_SIZE])
screen.fill(WHITE)
pygame.display.set_caption("Life")


clock = pygame.time.Clock()


isEditing = False
mouseX = -1; mouseY = -1

running = True  
while running:
  clock.tick(60)

  # handle events
  for event in pygame.event.get():
    if event.type == QUIT: 
      running = False
  
    if event.type == KEYDOWN:
      if event.key == K_e: 
        isEditing = True 
      elif event.key == K_x:
        isEditing = False

    if event.type == MOUSEBUTTONDOWN and isEditing:
      mouseX, mouseY = pygame.mouse.get_pos()


  # update
  if isEditing:
    if mouseX != -1:
      toggleCell(mouseX, mouseY)
      mouseX = -1; mouseY = -1
  else:
    updateGrid()

  # redraw
  screen.fill(WHITE)  
  drawGrid(screen);
  pygame.display.update()

pygame.quit()

